import logging
from pathlib import Path
from typing import Optional

from bx_py_utils.path import assert_is_dir
from rich import print

from manageprojects.cookiecutter_api import execute_cookiecutter
from manageprojects.data_classes import GenerateTemplatePatchResult
from manageprojects.git import Git
from manageprojects.utilities.temp_path import TemporaryDirectory


logger = logging.getLogger(__name__)


def generate_template_patch(
    *,
    project_path: Path,
    template: str,  # CookieCutter Template path or GitHub url
    directory: str = None,  # Directory name of the CookieCutter Template
    from_rev: str,
    replay_context: dict,
    password: str = None,
    config_file: Optional[Path] = None,  # Optional path to 'cookiecutter_config.yaml'
    cleanup: bool = True,  # Remove temp files if not exceptions happens
) -> Optional[GenerateTemplatePatchResult]:
    """
    Create git diff/patch from cookiecutter template changes.
    """
    print(f'Generate update patch for project: {project_path} from {template}')
    project_name = project_path.name

    with TemporaryDirectory(prefix=f'manageprojects_{project_name}_', cleanup=cleanup) as temp_path:

        #############################################################################
        # Generate the cookiecutter template in the current/HEAD version:

        compiled_to_path = temp_path / 'to_rev_compiled'
        print(f'Compile cookiecutter template in the current version here: {compiled_to_path}')
        cookiecutter_context, destination_path, to_rev_repo_path = execute_cookiecutter(
            template=template,
            directory=directory,
            output_dir=compiled_to_path,
            no_input=True,
            extra_context=replay_context,
            # replay=replay,
            checkout=None,  # Checkout HEAD/main revision
            password=password,
            config_file=config_file,
        )
        assert_is_dir(to_rev_repo_path)
        assert destination_path.parent == compiled_to_path

        #############################################################################
        # Get the current git commit hash and date:

        git = Git(cwd=to_rev_repo_path, detect_root=True)
        to_rev = git.get_current_hash(verbose=False)
        to_commit_date = git.get_commit_date(verbose=False)
        print(f'Update from rev. {from_rev} to rev. {to_rev} ({to_commit_date})')

        if from_rev == to_rev:
            print(
                f'Latest version {from_rev!r}'
                f' from {to_commit_date} is already applied.'
                ' Nothing to update, ok.'
            )
            return None

        patch_file_path = Path(
            project_path, '.manageprojects', 'patches', f'{from_rev}_{to_rev}.patch'
        )
        print(f'Generate patch file: {patch_file_path}')

        #############################################################################
        # Generate the cookiecutter template in the old version:

        compiled_from_path = temp_path / f'{from_rev}_compiled'
        print(
            'Compile cookiecutter template in the'
            f' old {from_rev} version here: {compiled_from_path}'
        )
        cookiecutter_context, destination_path, from_repo_path = execute_cookiecutter(
            template=template,
            directory=directory,
            output_dir=compiled_from_path,
            no_input=True,
            extra_context=replay_context,
            # replay=replay,
            checkout=from_rev,  # Checkout the old revision
            password=password,
            config_file=config_file,
        )
        assert_is_dir(from_repo_path)
        assert from_repo_path == to_rev_repo_path
        assert destination_path.parent == compiled_from_path

        #############################################################################
        # Generate git patch between old and current version:

        patch = git.diff(compiled_from_path, compiled_to_path)
        if not patch:
            logger.warning(f'No gif diff between {from_rev} and {to_rev} !')
            print(f'No gif diff between {compiled_from_path} and {compiled_to_path} !')
            return None

        from_path_str = f'a{compiled_from_path}/'
        assert from_path_str in patch, f'{from_path_str!r} not found in patch: {patch}'
        patch = patch.replace(from_path_str, 'a/')

        to_path_str = f'b{compiled_to_path}/'
        assert to_path_str in patch, f'{to_path_str!r} not found in patch: {patch}'
        patch = patch.replace(to_path_str, 'b/')

        logger.info('Write patch file: %s', patch_file_path)
        patch_file_path.parent.mkdir(parents=True, exist_ok=True)
        patch_file_path.write_text(patch)
        return GenerateTemplatePatchResult(
            repo_path=to_rev_repo_path,  # == from_repo_path
            patch_file_path=patch_file_path,
            from_rev=from_rev,
            compiled_from_path=compiled_from_path,
            to_rev=to_rev,
            to_commit_date=to_commit_date,
            compiled_to_path=compiled_to_path,
        )
