# AUTOGENERATED! DO NOT EDIT! File to edit: 00_core.ipynb (unless otherwise specified).

__all__ = ['sparse_mask', 'masked_params', 'apply_masks', 'is_sparseable_module', 'sparseable_modules',
           'uniform_sparsity', 'first_layer_dense_uniform', 'erdos_renyi_sparsity', 'sparsify_model',
           'weight_magnitude', 'gradient_magnitude', 'gradient_momentum', 'top_k_mask', 'DynamicSparseTrainingCallback',
           'SET_kwargs', 'STFS_kwargs', 'RigL_kwargs']

# Cell
import numpy as np
import torch
import torch.nn as nn

# Cell
from fastcore.all import * # L
from fastai.basics import * # flatten_model, etc...

# Cell
@torch.no_grad()
def sparse_mask(sizes, sparsity):
    n_total = np.prod(sizes)
    n_ones = round((1-sparsity) * n_total)
    shuffled_ones = torch.randperm(n_total)[:n_ones]
    mask = torch.zeros(n_total, dtype=torch.bool)
    mask[shuffled_ones] = True
    return mask.reshape(*sizes)

# Cell
def masked_params(module):
    '''Returns list of (param, mask) tuples, assuming masks are buffers with name scheme: {param}_mask.'''
    buffer_d = {name:b for name, b in module.named_buffers()}
    param_mask_pairs = [(p, buffer_d[f'{name}_mask'])
                        for name,p in module.named_parameters()
                        if f'{name}_mask' in buffer_d]
    return param_mask_pairs

# Cell
@torch.no_grad()
def apply_masks(module, *args, inplace=True):
    for p, mask in masked_params(module):
        if inplace: p.data.mul_(mask)
        else:       p.data = p.data.mul(mask)

# Cell
_sparseable_module_types = nn.Linear, nn.Conv2d

def is_sparseable_module(m, additional_types=[]):
    types = set(_sparseable_module_types)
    if additional_types: types |= set(additional_types)
    return isinstance(m, tuple(types))

# Cell
def sparseable_modules(model, additional_types=[]):
    return [m for m in flatten_model(model) if is_sparseable_module(m, additional_types)]

# Cell
def uniform_sparsity(params, model_sparsity):
    return [model_sparsity] * len(params)

# Cell
def first_layer_dense_uniform(params, model_sparsity):
    sparsities = [1.] + [model_sparsity] * (len(params) - 1)
    return sparsities

# Cell
# modified from https://github.com/google-research/rigl/blob/master/rigl/sparse_utils.py.
def erdos_renyi_sparsity(params, model_sparsity, include_kernel=True, erk_power_scale=1.0):
    """
    Returns a list of sparsities in the same order as params. Sparsities satisfy
    the Erdos-Renyi(Kernel) distribution, where the model has a total parameter count
    as one with uniform sparsities, that is, satisfying the following equation:
    # eps * (p_1 * N_1 + p_2 * N_2) = (1 - model_sparsity) * (N_1 + N_2), for some float `eps`.

    Args:
    params: list of all sparseable parameters
    model_sparsity: target overall sparsity between 0 and 1
    include_kernel: if True, kernel dimensions are included in the scaling (e.g. for ConvNd layers)
    erk_power_scale: scale < 1 softens the erdos_renyi distribution (i.e. closer to uniform)

    Returns a list of sparsities where values correspond to individual param sparsities.
    """
    # Enforce custom sparsities, then find correct scaling factor, `eps` for remaining params
    dense_layers = set()
    is_eps_valid = False
    while not is_eps_valid:
        # Start with all layers and try to find right eps. If any sparsity exceeds 1,
        # make that layer dense and repeat with the non-dense layers.
        #
        # E.g. where N_3, and N_4 are found to be dense:
        # eps * (p_1 * N_1 + p_2 * N_2) + (N_3 + N_4) =
        #    (1 - model_sparsity) * (N_1 + N_2 + N_3 + N_4)
        # eps * (p_1 * N_1 + p_2 * N_2) =
        #    (1 - model_sparsity) * (N_1 + N_2) - model_sparsity * (N_3 + N_4) <--- == rhs
        # eps = rhs / (\sum_i p_i * N_i) <--- == divisor
        # eps = rhs / divisor

        divisor = 0
        rhs = 0
        raw_sparsity = {}
        for p in params:
            n_zeros = int(np.floor(model_sparsity * p.numel()))
            if p in dense_layers:
                rhs -= n_zeros
            else:
                n_ones = p.numel() - n_zeros
                rhs += n_ones
                if include_kernel:
                    raw_sparsity[p] = (np.sum(p.shape) / np.prod(p.shape))**erk_power_scale
                else:
                    raw_sparsity[p] = (np.sum(p.shape[:2]) / np.prod(p.shape[:2]))
                divisor += raw_sparsity[p] * p.numel()

        eps = rhs / divisor

        # If eps * raw_sparsity[p] > 1, we add the param to the set of dense_layers
        max_sparsity = np.max(list(raw_sparsity.values()))
        if eps * max_sparsity > 1:
            for p, p_raw_sparsity in raw_sparsity.items():
                if p_raw_sparsity == max_sparsity:
                    dense_layers.add(p)
        else:
            is_eps_valid = True

    # With the valid eps, we can set sparsities of the remaining layers
    sparsities = [0. if p in dense_layers else (1. - eps * raw_sparsity[p]) for p in params]
    return sparsities

# Cell
@torch.no_grad()
def sparsify_model(model, model_sparsity, sparse_init_f=uniform_sparsity, enforce_mask=True):
    '''
    Adds a sparse mask for each sparseable-module weight in model and applies mask to weights.

    If `enforce_mask` is True, a forward_pre_hook will be registered to each module
    to apply the weight mask before every forward pass of the module.

    `sparsify_method`: per RigL paper, `uniform_sparsity` has fewer FLOPs, `erdos_renyi_sparsity`
    results in better model.

    Returns hooks if `enforce_mask` == True, otherwise None.
    '''
    sparseable_modules = L(model.modules()).filter(is_sparseable_module)
    sparseable_params = sparseable_modules.map(lambda m: m.weight)
    sparsities = sparse_init_f(sparseable_params, model_sparsity)
    hooks = []
    for m, s in zip(sparseable_modules, sparsities):
        if s > 0:
            mask = sparse_mask(m.weight.shape, s).to(m.weight.device)
            m.register_buffer('weight_mask', mask)
            apply_masks(m)
            if enforce_mask:
                h = m.register_forward_pre_hook(apply_masks)
                hooks.append(h)
    return hooks or None

# Cell
def weight_magnitude(p, *args): return p.data.abs()

# Cell
def gradient_magnitude(p, *args): return p.grad.abs()

# Cell
def gradient_momentum(p, opt, *args):
    '''Calculates the momentum of the gradient for a parameter `p` from the `opt` state.'''
    state = opt.state[p]
    grad_avg = state['grad_avg'] if 'grad_avg' in state else None
    sqr_avg = state['sqr_avg'] if 'sqr_avg' in state else None
    if grad_avg is None:
        raise Exception(f"Error: 'grad_avg' key not found in optimizer state. Tip: set the `mom` hyperparamter in the learner.")
    if sqr_avg is None:
        grad_mom = grad_avg
    else:
        try: eps = opt.state_dict()['hypers'][0]['eps']
        except: eps = 1e-6
        print(eps)
        grad_mom =  grad_avg / (torch.sqrt(sqr_avg + eps))
    return grad_mom

# Cell
def top_k_mask(t, n_keep):
    '''Returns a mask with `n_keep` ones cooresponding to the largest values in `t`'''
    n_drop = t.numel() - n_keep
    _, sorted_ixs = torch.topk(t.flatten(), k=t.numel())
    mask = torch.cat([torch.ones(n_keep, dtype=torch.bool, device=t.device),
                      torch.zeros(n_drop, dtype=torch.bool, device=t.device)])
    mask = mask.scatter(0, sorted_ixs, mask)
    return mask.view(*t.shape)

# Cell
class DynamicSparseTrainingCallback(Callback):
    toward_end = True # run after GradientAccumulation and any other cb that modifies the gradients
    _exclude_modules = [nn.BatchNorm1d, nn.BatchNorm2d, nn.BatchNorm3d]
    _sparse_distributions = ('uniform', 'ERK')

    def __init__(self, sparsity=0.9, modules=None, sparsity_distribution='uniform',
                 batches_per_update=None,
                 initial_drop_grow_pct=0.3, stop_pct=0.75,
                 keep_method=weight_magnitude,
                 grow_method=gradient_magnitude,
                 exclude_modules=[], first_layer_dense=True):
        store_attr('sparsity,modules,sparsity_distribution,initial_drop_grow_pct,stop_pct,keep_method,grow_method,first_layer_dense')
        self.batches_per_update = ifnone(batches_per_update, len(self.dls.train)) # default: 1 update per epoch
        self.exclude_modules = exclude_modules + self._exclude_modules

    def before_fit(self):
        ### determine modules to sparsify
        is_sparse_module = lambda m: has_params(m) and hasattr(m, 'weight') and type(m) not in self.exclude_modules
        self.modules = ifnone(self.modules, [m for m in flatten_model(self.learn.model) if is_sparse_module(m)])

        ### determine initial sparsities per layer
        assert self.sparsity_distribution in self._sparse_distributions, f'Unknown sparsity distribution: {self.sparsity_distribution}. Options: {self._sparse_distributions}'
        if self.sparsity_distribution == 'uniform':
            self.S = [self.sparsity] * len(self.modules)
            if self.first_layer_dense:
                self.S[0] = 0
        elif self.sparsity_distribution == 'ERK':
            raise NotImplementedError()

        if self.grow_method not in self._grow_methods: assert is_function(self.grow_method)

        ### create masks and assign to each parameter
        for m, s in zip(self.modules, self.S):
            if s > 0: m.register_buffer('weight_mask', (torch.rand_like(m.weight) > s).bool())

        ### schedule the decay percent (i.e. # of connections to drop/add per update)
        self.drop_grow_pct_sched = combine_scheds([self.stop_pct, 1-self.stop_pct],
                                                  [SchedCos(self.initial_drop_grow_pct, 0.), SchedNo(0.,0.)])

        ### apply weight masks
        self.add_hooks()

    def after_fit(self):
        self.remove_hooks() # ensure hooks are removed (e.g. in case we cancelled the fit loop)

    def add_hooks(self):
        self.remove_hooks() # to be certain that we never add them twice
        self.hooks = [m.register_forward_pre_hook(apply_masks) for m in self.modules]

    def remove_hooks(self):
        if getattr(self, 'hooks', None):
            for h in self.hooks: h.remove()

    def before_batch(self):
        if not self.training: return
        if self.is_update_step():
#             print(f'UPDATE step! before_batch, epoch: {self.learn.epoch}, step: {self.learn.iter}')
            self.hooks = Hooks(self.modules, self.rewire_module, is_forward=False)
    def after_backward(self):
        if self.is_update_step():
#             print(f'UPDATE step! after_backward, epoch: {self.learn.epoch}, step: {self.learn.iter}')
            self.hooks.remove()

            ### skip gradient update after changing network connectivity
            raise CancelBatchException()

    def is_update_step(self):
        '''Whether to modify network connectivity. Side effect: updates self.drop_grow_pct'''
        step = self.epoch * self.n_iter + self.iter
        n_steps = self.n_epoch * self.n_iter
        pct_train = step / n_steps
        self.drop_grow_pct = self.drop_grow_pct_sched(pct_train)
        return step > 0 and step % self.batches_per_update == 0 and self.drop_grow_pct > 0

    @torch.no_grad()
    def rewire_module(self, m, *args):
        '''Update step for one module'''

        for m, s in zip(self.modules, self.S):
            if s <= 0: continue # ignore fully dense layers

            param, mask = m.weight, m.weight_mask

            ### determine # of connections to keep, # to regrow
            n_keep = self.compute_n_keep(s, param, mask)
            n_grow = self.compute_n_grow(s, param, mask)

            ### determine weights to keep
            keep_score = self.keep_method(p)
            keep_mask = top_k_mask(keep_score, n_keep)

            ### determine weights to grow
            grow_score = self.grow_method(p, self.learn.opt)
            # set keep weights to negative so we don't choose to grow them
            grow_score = grow_score * keep_mask.logical_not() - keep_mask.float()
            grow_mask = top_k_mask(grow_score, n_grow)

            ### update mask
            mask.data = keep_mask | grow_mask

            ### zero momentum for new connections
            self.reset_momentum(param, grow_mask & keep_mask.logical_not())

    def compute_n_grow(self, s, p, mask):
        return (1 - s) * p.numel() * self.drop_grow_pct
    def compute_n_keep(self, s, p, mask):
        return p.numel() - self.compute_n_grow()

    @torch.no_grad()
    def reset_momentum(self, p, grow_mask):
        '''Initialize momentum to zero for newly-added connections'''
        state = self.opt.state[p]
        if 'grad_avg' in state: state['grad_avg'].mul_(grow_mask)
        if 'sqr_avg' in state: state['sqr_avg'].mul_(grow_mask)

#     _docs = dict(before_fit="Set counter to 0",
#                  after_backward="Skip weight update if we have not seen enough items")

# Cell
SET_kwargs = {}

# Cell
STFS_kwargs = {'keep_method': weight_magnitude, 'grow_method': gradient_momentum,
               'batches_per_update': None, 'initial_drop_grow_pct': 0.3, 'stop_pct': 1.0,}

# Cell
RigL_kwargs = {'keep_method': weight_magnitude, 'grow_method': gradient_magnitude,
               'batches_per_update':None, 'initial_drop_grow_pct':0.3, 'stop_pct':0.75,}